#!/usr/bin/env bash
#
# 3.0.0.1	- 07/08/2020 Linux v3
# 3.0.0.2	- 19/01/2021 Change ua-tool for ua-tool-support
#			     Eliminated ua-config call
# 3.1.0.0   	-08-02-2022. New features:
#               	Activation/deactivation kernel module and watchdog traces
#               	Launch specific version of the ua-tool-support, depending of
#                  	  system architechture(x86/x64) and master type(DEPS/NODEPS)
#               	If commands execution fails, redirect stderr to log file
#               	Added product_detected.txt file if master is deps or not deps
#			Changed psanhotsd log level from debug to TRACE
# 3.1.0.1	- 01/06/2022
#
# 3.1.0.2	- 15/06/2022
#			Watchdog traces are only disabled if the log is NOT empty
#			Script must be run with administrator privileges
#
# 3.2.1.0	- New Script for R14: 3.02.01.0002
#			Fix for putty columns resizing
#			New folders messages/queue/quarantine/reports added
#			Added pstree, hostnamectl, lslogins, lsipc and pa_cmd commands
#			Product files list, hashes and dependencies
#			Several minor changes
#
# 3.2.1.1	- Reverted to env bash
#
# 3.3.0.1	- New Script for R15: 3.03.00.0001
#			Added compatibility for R14 and R15 and a lot of other changes
# 3.5.0.1	- New Script for R16: 3.05.00.0001
#			Added compatibility for R16 and a lot of other changes
# 3.6.0.0	- New Script for R17: 3.06.00.0001
#			Added compatibility for R17 and a lot of other changes


version="3.6.0.0"

# For putty or terminal emulator
export COLUMNS=500

OS=$(uname -s)
ID=$(id -u)


MA_DIR="/usr/local/management-agent/"
PA_DIR="/opt/panda-security/endpoint/"
PA_VAR_DIR="/var/opt/panda-security/endpoint/"

[ -f $PA_DIR"/bin/psanhostd" ] && PA_INSTALLED=1 || PA_INSTALLED=0
[ -f "/usr/local/management-agent/bin/AgentSvc" ] && MA_INSTALLED=1 || MA_INSTALLED=0
[ -f "/opt/CentraStage/agent.sh" ] && CS_INSTALLED=1 || CS_INSTALLED=0


LOGS_DIR="aether-logs-$(/bin/date +%Y%m%d-%H%M%S/)$OS"
LOGS_ROOT_DIR="aether-logs-$(/bin/date +%Y%m%d-%H%M%S/)"

CORES_DIR="$LOGS_DIR/cores/"

MD5SUM="md5sum"

URLS="8.8.8.8 www.google.es iext.pandasecurity.com cpp-kw.pandasecurity.com cpp-fulg.pandasecurity.com cpp-fusm.pandasecurity.com"

CURL_AVAILABLE=$(curl -V &>/dev/null; echo $?)


function is_64bits_architecture
{
	arch=$(od -An -t x1 -j 4 -N 1 $PA_DIR"/bin/psanhostd")
		
	if [[ $arch -eq 2 ]]; then
		echo "0"
	else
		echo "1"
	fi
}

IS_X64_PROTECTION=$(is_64bits_architecture)


function is_linux
{
	if [ "$OS" = "Linux" ] ; then
		echo "$OS Found!"
	else
		echo "OS not supported? -> $OS"
		exit 1
	fi
}


function is_administrator
{
	if [ $UID -ne 0 ]; then
		echo "User must be root to run this script"
		exit 1
	fi
}


function get_distribution()
{
	if [ -f /etc/os-release ]
	then
		source /etc/os-release &> /dev/null
		echo "$ID"
	else
		DISTRI_INFO=$(cat /etc/*-release)

		if echo $DISTRI_INFO | grep -q "Red Hat"
		then
			echo "rhel"
		elif echo $DISTRI_INFO | grep -q "CentOS"
		then
			echo "centos"
		else
			echo ""
		fi
	fi
}


linux_dist=$(get_distribution)
UAVersion=$(cat /opt/panda-security/endpoint/buildinfo.txt | grep Version | cut -d ' ' -f 4)

if cat $PA_DIR/buildinfo.txt | grep 03.05. > /dev/null; then
	UA_TOOL=$PA_DIR"/internal_tools/R16/ua-tool-support"
elif cat $PA_DIR/buildinfo.txt | grep 03.06. > /dev/null; then
	UA_TOOL=$PA_DIR"/internal_tools/R17/ua-tool-support"
elif cat $PA_DIR/buildinfo.txt | grep 03.07. > /dev/null; then
	UA_TOOL=$PA_DIR"/internal_tools/R18/ua-tool-support"
else
	UA_TOOL=$PA_DIR""
fi



if [[ $IS_X64_PROTECTION -eq 0 ]]; then
	UA_TOOL=$UA_TOOL"-x64"
else
	UA_TOOL=$UA_TOOL"-x86"
fi

ld_master_no_deps=$PA_DIR"/lib/ld-linux*.so"
ls -lh $ld_master_no_deps &> /dev/null

if [ $? -eq 0 ]; then
	UA_TOOL=$UA_TOOL"-nodeps"
else
	UA_TOOL=$UA_TOOL"-deps"
fi

if [[ ! -f $UA_TOOL ]]; then
	echo -e "[!] It won't be possible to enable Protection traces or check the status of processes."
elif [[ ! -x $UA_TOOL ]]; then
	chmod 744 $UA_TOOL
fi


function get_product_mode
{
	ld_master_no_deps=$PA_DIR"/lib/ld-linux*.so"
	ls -lh $ld_master_no_deps &> /dev/null

	if [[ $? -eq 0 ]]; then
		echo "Instalación sin dependencias (NODEPS)" > $LOGS_DIR"/pa/product_mode.txt"
	else
		echo "Instalación con dependencias (DEPS)" > $LOGS_DIR"/pa/product_mode.txt"
	fi
}


function gather_abrtd_cores
{
	for i in $( find $1 -name 'cmdline' -exec grep -ilE "(psanhostd|PavapiDaemon|AgentSvc)" {} \;)
	do
		echo $i:$( cat $i) >> $CORES_DIR/core_list.txt
		cp -a $(dirname $i) $CORES_DIR
	done
}


function gather_linux_cores
{
	for f in $(find $1 -maxdepth 2 -type $2)
	do
		if echo $f | grep -ilE "(psanhostd|PavapiDaemon|AgentSvc)"; then
			cp -a $f $CORES_DIR/
		fi
	done
}


function check_product_installed 
{
    if [ ! -d $PA_DIR ]
    then
        echo "[+]"
	echo "[!] $PA_DIR does not exists, is Protection installed?"
        echo
        return
    fi
}


function gather_linux_logs 
{
	echo "[+] Gathering system logs..."
	echo "[+]"
	
	echo "[+] Launching 30 iterations of top command, this will take some time. Please wait 90 seconds..."
	{ export COLUMNS=500; top -c -n 30 -b; } > $LOGS_DIR/top-30.txt 2>&1
	echo "[+]"
	echo "[+] Continuing with data collection..."
	
	{ export COLUMNS=500; ps fuxa; } > $LOGS_DIR/ps.txt 2>&1
	{ export COLUMNS=500; pstree; printf "\n\n\n"; pstree -alU; } > $LOGS_DIR/pstree.txt 2>&1
	{ lsof -i -n -P +c15;  printf "\n\n"; ss -A all -m -e -p -i -t -u; } > $LOGS_DIR/netstat.txt 2>&1
	{ free -m; printf "\n\n"; cat /proc/meminfo; } > $LOGS_DIR/free.txt 2>&1
	uname -a >& $LOGS_DIR/uname.txt
	more /etc/*release* | cat >& $LOGS_DIR/etc_releases.txt
	lsb_release -a >& $LOGS_DIR/lsb_release.txt
	hostnamectl >& $LOGS_DIR/hostnamectl.txt
	lsipc >& $LOGS_DIR/lsipc.txt
	lsmod >& $LOGS_DIR/lsmod.txt
	iostat -d 1 10 >& $LOGS_DIR/iostat-10.txt 
	ip r show table all >& $LOGS_DIR/routes.txt || route -n >& $LOGS_DIR/routes.txt || cat /proc/net/route >& $LOGS_DIR/routes.txt
	{ iptables -t nat -L -v;  printf "\n\n"; iptables -t mangle -L -v; printf "\n\n"; nft list ruleset; } >& $LOGS_DIR/iptables.txt
	lshw >& $LOGS_DIR/lshw.txt 
	vmstat -wn 1 10 >& $LOGS_DIR/vmstat.txt 
	mokutil --sb-state >& $LOGS_DIR/secureboot.txt 
	ls /sys/firmware/efi/ >> $LOGS_DIR/secureboot.txt 2>&1
	cat /proc/cpuinfo >& $LOGS_DIR/cpuinfo.txt
	{ last -n 5 -R; printf "\n\n"; uptime; printf "\n\n"; lslogins; } >& $LOGS_DIR/last-5.txt
	df -h >& $LOGS_DIR/df.txt
	sysctl -a >& $LOGS_DIR/sysctl.txt 2>/dev/null
	ifconfig -a >& $LOGS_DIR/ip_a.txt || ip a >& $LOGS_DIR/ip_a.txt
	ulimit -a >& $LOGS_DIR/ulimit.txt
	{ mount; printf "\n\n"; cat /etc/mtab; } > $LOGS_DIR/mount.txt 2>&1
        {
                printf ">>>>> Output of: fips-mode-setup --check <<<<<\n"
                fips-mode-setup --check
                printf "\n\n>>>>> Content of: cat /proc/sys/crypto/fips_enabled <<<<<\n"
                cat /proc/sys/crypto/fips_enabled
                printf "\n\n>>>>> Output of: sestatus -v <<<<<\n"
                sestatus -v
                printf "\n\n>>>>> Output of: ps -eZ | grep -i 'psanhostd\|AgentSvc\|PavapiDaemon\|psanwatchdog' <<<<<\n"
                ps -eZ | grep -i 'psanhostd\|AgentSvc\|PavapiDaemon\|psanwatchdog'
                printf "\n\n>>>>> Output of: update-crypto-policies --show <<<<<\n"
                update-crypto-policies --show
        } > $LOGS_DIR/fips.txt 2>&1

	dmesg -T >& $LOGS_DIR/dmesg.txt
	if [[ $? -ne 0 ]]; then
		dmesg >& $LOGS_DIR/dmesg.txt
	fi

	journalctl -a >& $LOGS_DIR/journalctl.txt
	
	if [ -d /var/spool/abrt ]; then
		gather_abrtd_cores "/var/spool/abrt"
	fi
	if [ -d /var/tmp/abrt ]; then
		gather_abrtd_cores "/var/tmp/abrt"
	fi
	if [ -d /var/lib/systemd/coredump ]; then
		gather_linux_cores "/var/lib/systemd/coredump" "d"
	fi
	if [ -d /var/crash ]; then
		gather_linux_cores "/var/crash" "f"
	fi

	coredumpctl >& $CORES_DIR/coredumpctl.txt
	coredumpctl -1 >> $CORES_DIR/coredumpctl.txt 2>&1

	if [ "$(ls -A /var/crash/)" ]; then
        	cp -p "/boot/System.map-$(uname -r)" "$CORES_DIR"
        	cp -p "/boot/vmlinuz-$(uname -r)" "$CORES_DIR"
		ls -lR /var/crash/ >& $CORES_DIR/var_crash.txt
		find /var/crash/ -type f -name "vmcore-dmesg*" | while read -r file; do
			mkdir -p "$CORES_DIR/var/crash/$(dirname ${file#/var/crash/})"
			cp -p "$file" "$CORES_DIR/var/crash/$(dirname ${file#/var/crash/})"
		done
	else
		echo "/var/crash/ is empty." >& $CORES_DIR/var_crash.txt
	fi

	if [ "$FLAVOUR" = "ubuntu" -o "$FLAVOUR" = "debian" -o "$FLAVOUR" = "linuxmint" ]; then
		dpkg -l > $LOGS_DIR/dpkg.txt || true
		cp /etc/apt/sources.list.d/panda* $LOGS_DIR
	elif [ "$FLAVOUR" = "sles" -o "$FLAVOUR" = "sled" -o "$FLAVOUR" = "opensuse-leap" -o "$FLAVOUR" = "sle_hpc" -o "$FLAVOUR" = "sles_sap" ]; then
		zypper --non-interactive se -s management-agent protection-agent endpointprotection >& $LOGS_DIR/zypper.txt
		rpm -qa > $LOGS_DIR/rpm.txt || echo "[-] no rpm package on a non Debian based distribution"
		cp /etc/yum.zypp.d/panda* $LOGS_DIR
	else
		yum list installed | xargs -n3 | column -t >& $LOGS_DIR/yum.txt		
		rpm -qa > $LOGS_DIR/rpm.txt || echo "[-] no rpm package on a non Debian based distribution"
		cp /etc/yum.repos.d/panda* $LOGS_DIR

                if [ "$(ls -A /var/lib/yum/repos/)" ]; then
                        find /var/lib/yum/repos/ -type d | grep -E "(management-agent|protection-agent)" >& $LOGS_DIR/yum_repos_cache.txt
                else
                        echo "/var/lib/yum/repos/ is empty." >& $LOGS_DIR/yum_repos_cache.txt
                fi

	fi


	for f in $(seq 0 ${#LOGS[*]})
	do
		if [ "${LOGS[$f]}" != "" ]; then
#			echo "[+] Copying ${LOGS[$f]} to $LOGS_DIR/"
			cp -a "${LOGS[$f]}" $LOGS_DIR/ &>/dev/null
		fi
	done
}


function gather_pa_logs
{
	if [ $PA_INSTALLED -eq 1 ]
	then
		mkdir -p $LOGS_DIR/pa/
		for f in $(seq 0 ${#PA_LOGS[*]})
		do
			if [ "${PA_LOGS[$f]}" != "" ]; then
#				echo "[+] Copying ${PA_LOGS[$f]} to $LOGS_DIR/pa/"
				cp -a "${PA_LOGS[$f]}" $LOGS_DIR/pa/ &>/dev/null
			fi
		done

		ldd $PA_DIR/bin/* $PA_DIR/lib/* $PA_DIR/lib32/* $PA_DIR/pavapi/daemon/* $PA_DIR/pavapi/pavapi/* > $LOGS_DIR/pa/PA_files_ldd.txt 2>&1
		{ ls -l /proc/`pgrep psanhostd`/fd; printf "\n\n"; ls -l /proc/`pgrep PavapiDaemon`/fd; } > $LOGS_DIR/pa/PA_files_fd.txt 2>&1
		find $PA_DIR -type f -exec md5sum '{}' \; > $LOGS_DIR/pa/PA_files_md5.txt 2>&1
		find $PA_VAR_DIR -type f -exec md5sum '{}' \; >> $LOGS_DIR/pa/PA_files_md5.txt 2>&1
		ls -lR $PA_DIR > $LOGS_DIR/pa/PA_files_tree.log 2>&1
		ls -lR $PA_VAR_DIR >> $LOGS_DIR/pa/PA_files_tree.log 2>&1

		{ systemctl status endpoint-protection.service; printf "\n\n"; service endpoint-protection.service status; printf "\n\n"; status endpoint-protection.service; } > $LOGS_DIR/pa/service_status.txt 2>&1
		
		{
			printf ">>>>> Output of: modinfo protection_agent (installed module) <<<<<\n"
			modinfo protection_agent
			printf "\n\n>>>>> Content of: cat /sys/module/protection_agent/version (really loaded module) <<<<<\n"
			cat /sys/module/protection_agent/version
		} > $LOGS_DIR/pa/modinfo.txt 2>&1

		cat /proc/slabinfo >& $LOGS_DIR/pa/slabinfo.txt
		
		find /lib/modules/ -name protection_agent.ko -exec md5sum {} \; >& $LOGS_DIR/pa/lib_modules.txt
		{ printf "\n\n"; ls -lR /lib/modules/; } >> $LOGS_DIR/pa/lib_modules.txt 2>&1

		dmesg -T >& $LOGS_DIR/pa/protection_agent.log
		if [[ $? -ne 0 ]]; then
			dmesg | grep -i protection-agent >& $LOGS_DIR/pa/protection_agent.log
		else
			dmesg -T | grep -i protection-agent >& $LOGS_DIR/pa/protection_agent.log
		fi
		
		cp $PA_DIR/buildinfo.txt $LOGS_DIR/pa/
	
		{ > $PA_VAR_DIR/log/pa_cmd.log; $PA_DIR/bin/pa_cmd --info; } > $LOGS_DIR/pa/pa_cmd.txt 2>&1

		if [ "$(ls -A /var/opt/panda-security/endpoint/cache/)" ]; then
			strings /var/opt/panda-security/endpoint/cache/*.DAT | grep VERSION >& $LOGS_DIR/pa/EvtDet.txt
		else
			echo "/var/opt/panda-security/endpoint/cache/ is empty." >& $LOGS_DIR/pa/EvtDet.txt
		fi
		

		PID=$(pgrep psanhostd)
		[ "$PID" != "" ] && lsof -a -p $PID >& $LOGS_DIR/pa/lsof_psanhostd.txt
		PID=$(pgrep psanwatchdog)
		[ "$PID" != "" ] && lsof -a -p $PID >& $LOGS_DIR/pa/lsof_psanwatchdog.txt
		PID=$(pgrep PavapiDaemon)
		[ "$PID" != "" ] && lsof -a -p $PID >& $LOGS_DIR/pa/lsof_PavapiDaemon.txt
	
		get_product_mode

		if [[ -f $UA_TOOL ]]
		then
			echo -e "[+] Gathering Protection info/status/reports/connectivity..."
			echo "[+]"
			$UA_TOOL --info >& $LOGS_DIR/pa/uatool_info.txt
			$UA_TOOL --status >& $LOGS_DIR/pa/uatool_status.txt
			if $UA_TOOL | grep "R16"; then
				$UA_TOOL --connectivity >& $LOGS_DIR/pa/uatool_comms.txt
			else
				$UA_TOOL --communications --connectivitytest >& $LOGS_DIR/pa/uatool_comms.txt
			fi
			$UA_TOOL --report --showall >& $LOGS_DIR/pa/uatool_reports.txt
			$UA_TOOL --quarantine --all >& $LOGS_DIR/pa/uatool_quarantine.txt
		else
			echo -e "[+] Gathering Protection info/status/reports/connectivity..."
			echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
			echo "[+]"
		fi

	else
		echo "[-] Protection not installed, skipping..."
		echo "[+]"
	fi
}


function gather_ma_logs
{
	if [ $MA_INSTALLED -eq 1 ]
	then
		mkdir -p $LOGS_DIR/ma/
		for f in $(seq 0 ${#MA_LOGS[*]})
		do
			if [ "${MA_LOGS[$f]}" != "" ]; then
#				echo "[+] Copying ${MA_LOGS[$f]} to $LOGS_DIR/ma/"
				cp -a "${MA_LOGS[$f]}" $LOGS_DIR/ma/ &>/dev/null
			fi
		done

		ldd $MA_DIR/bin/* $MA_DIR/features/* $MA_DIR/lib/* $MA_DIR/plugins/* $MA_DIR/PatchManagement/* > $LOGS_DIR/ma/MA_files_ldd.txt 2>&1
		{ ls -l /proc/`pgrep AgentSvc`/fd; } > $LOGS_DIR/ma/MA_files_fd.txt 2>&1
		find $MA_DIR -type f -exec md5sum '{}' \; > $LOGS_DIR/ma/MA_files_md5.txt
		ls -lR $MA_DIR > $LOGS_DIR/ma/MA_files_tree.log 2>&1

		{ systemctl status management-agent.service; printf "\n\n"; service management-agent.service status; printf "\n\n"; status management-agent.service; } > $LOGS_DIR/ma/service_status.txt 2>&1

		PID=$(pgrep AgentSvc)
                [ "$PID" != "" ] && lsof -a -p $PID >& $LOGS_DIR/ma/lsof_AgentSvc.txt

	else
		echo "[-] Agent not installed, skipping..."
		echo "[+]"
	fi
}


function gather_cs_logs
{
	if [ $CS_INSTALLED -eq 1 ]
	then
		mkdir -p $LOGS_DIR/cs/
		cp -p --parents /opt/CentraStage/log.txt /opt/CentraStage/archives/* /usr/local/share/CentraStage/AEMAgent/DataLog/* /usr/local/share/CentraStage/Monitoring/*  $LOGS_DIR/cs/ 2>&1
        else
                echo "[-] CS not installed, skipping..."
                echo "[+]"
        fi
}


function gather_logs
{
	show_version
	echo

	echo "[+] $(is_linux)"
	echo "[+]"

	FLAVOUR=$(get_distribution)
	echo "[+] Linux flavour: $FLAVOUR!"

	check_product_installed


	REPOSITORIES="$MA_DIR/repositories"
	
	LOGS=(
		/var/log/syslog
		/var/log/messages
		/var/log/kern.log
		/var/log/debug
		/var/log/daemon
		/var/log/dnf.log
		/var/log/yum.log
		/var/log/dnf.rpm.log
		/etc/hosts
		/etc/hostname
		/etc/lsb-release
		/etc/issue
		/etc/resolv.conf
		/var/log/audit/audit.log
		/var/log/apt/history.log
		/etc/audit/audit.rules
	)

	PA_LOGS=($PA_VAR_DIR/log/ $PA_VAR_DIR/quarantine/ $PA_VAR_DIR/reports/ $PA_VAR_DIR/cfgdata/)
	MA_LOGS=($MA_DIR/log/ $MA_DIR/etc/ $MA_DIR/messages/ $MA_DIR/queue/ $MA_DIR/repositories/ $MA_DIR/downloads/KnowledgeFiles/)

	mkdir -p $LOGS_DIR || (echo "Logs directory not created, check for permissions or try again on another directory" && exit)
	mkdir -p $CORES_DIR
	

	disable_psanhostd_traces


	disable_kernel_traces


	echo "[+]"
	echo "[+] Gathering product logs and traces..."
	echo "[+]"


	command_name="lsof"
	
	if ! command -v $command_name >/dev/null 2>&1; then
    		echo "[!] '$command_name' command not found. Please, install '$command_name' package and run this script again."
		echo "[+]"
	fi


	gather_pa_logs


	gather_ma_logs


	gather_cs_logs


	gather_linux_logs


	if [[ $IS_X64_PROTECTION -eq 0 ]]; then
		if [[ -s $PA_VAR_DIR/log/psanwatchdog.log ]]; then
			disable_watchdog_traces
		fi
	fi


	echo "[+]"
	echo "[+] Almost finished. Organizing and compressing collected data..."
           

	if [[ $linux_dist != "sles" && $linux_dist != "sled" ]]; then
		for u in $URLS
		do
			ping -c 5 $u >> $LOGS_DIR/net_connectivity.txt 2>&1
		done
	fi
  
 
	chmod -R a+r $LOGS_DIR
	chown -R $(id -u -n) $LOGS_DIR
	TARFILE="$(hostname)_$version-$(date +%Y%m%d%H%M%S).tar.gz"
	tar zcf $TARFILE $LOGS_DIR/ || echo "[!!] targz archive not created. Please, compress $LOGS_DIR by any other method"
	rm -rf $LOGS_ROOT_DIR

	echo
	echo
	echo "[+] DONE!"
	echo 
	echo 
	echo "[+] Logs generated at `pwd`/$TARFILE"
	echo
}


function enable_psanhostd_traces
{
	delete_psanhostd_logs
	
	if [[ -f $UA_TOOL ]]; then
		echo "[+]"
		echo -e "[+] Enabling Protection Traces..."
#		$UA_TOOL --setloglevel --area ALL --level TRACE &> /dev/null
		$UA_TOOL --enabledefaultlogs &> /dev/null
		echo "[+]"
		echo "[+] Protection Traces Enabled!"
	else
		echo "[+]"
		echo -e "[+] Enabling Protection Traces..."
		echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
		echo "[+]"
	fi
}


function disable_psanhostd_traces
{
	if [[ -f $UA_TOOL ]]; then
		echo "[+]"
		echo -e "[+] Disabling Protection Traces..."
#		$UA_TOOL --setloglevel --area ALL --level OFF &> /dev/null
		$UA_TOOL --disablelogs &> /dev/null
		echo "[+]"
		echo "[+] Protection Traces Disabled!"
	else
		echo "[+]"
		echo -e "[+] Disabling Protection Traces..."
		echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
		echo "[+]"
	fi
}


function enable_kernel_traces
{
	if [[ -f $UA_TOOL ]]; then
		echo "[+]"
		echo -e "[+] Enabling Protection Kernel Module Traces..."
		$UA_TOOL --protectionagent --debugtraces "TRUE" &> /dev/null
		echo "[+]"
		echo "[+] Protection Kernel Module Traces Enabled!"
	else
		echo "[+]"
		echo -e "[+] Enabling Protection Kernel Module Traces..."
		echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
		echo "[+]"
	fi
}


function disable_kernel_traces
{
	if [[ -f $UA_TOOL ]]; then
		echo "[+]"
		echo -e "[+] Disabling Protection Kernel Module Traces..."
		$UA_TOOL --protectionagent --debugtraces "FALSE" &> /dev/null
		echo "[+]"
		echo "[+] Protection Kernel Module Traces Disabled!"
	else
		echo "[+]"
		echo -e "[+] Disabling Protection Kernel Module Traces..."
		echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
		echo "[+]"
	fi
}


function enable_watchdog_traces
{
	if [[ $IS_X64_PROTECTION -eq 1 ]]; then
		echo "[+]"
		echo -e "[-] Watchdog activation not available on 32bit RHEL6.x/CentOS6.x/OL6.x distributions"
		echo
		return
	fi

	delete_watchdog_logs
	
	if [[ -f $UA_TOOL ]]; then
		echo "[+]"
		echo -e "[+] Enabling Protection Watchdog Traces..."
		$UA_TOOL --watchdog --debugtraces "TRUE" &> /dev/null
		echo "[+]"
		echo "[+] Protection Watchdog Traces Enabled!"
	else
		echo "[+]"
		echo -e "[+] Enabling Protection Watchdog Traces..."
		echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
		echo "[+]"
	fi
}


function disable_watchdog_traces
{
	if [[ $IS_X64_PROTECTION -eq 1 ]]; then
		echo "[+]"
		echo -e "[-] Watchdog activation not available on 32bit RHEL6.x/CentOS6.x/OL6.x distributions"
		echo
		return
	fi

	if [[ -f $UA_TOOL ]]; then
		echo "[+]"
		echo -e "[+] Disabling Protection Watchdog Traces..."
		$UA_TOOL --watchdog --debugtraces "FALSE" &> /dev/null
		echo "[+]"
		echo "[+] Protection Watchdog Traces Disabled!"
	else
		echo "[+]"
		echo -e "[+] Disabling Protection Watchdog Traces..."
		echo -e "[!] Uncompatible tools for the installed Protection, skipping..."
		echo "[+]"
	fi
}


function disable_protection_traces
{
	disable_psanhostd_traces
	disable_kernel_traces
	disable_watchdog_traces
}


function delete_psanhostd_logs
{
	> $PA_VAR_DIR/log/psanhost*
	> $PA_VAR_DIR/log/ua-tool-support.log
}


function delete_watchdog_logs
{
	> $PA_VAR_DIR/log/psanwatchdog.log
}


function show_version
{
	echo
	echo "Script version: $version"
	echo 
}


function show_help
{
	echo 
	echo "Script to gather system and Universal Agent information."
	echo
	echo "Usage:"
	echo 
	echo -e "\t--help               Shows this help"
	echo -e "\t--version            Shows the script version"
	echo
	echo -e "\t--enable-protection  Enables protection daemon (psanhostd) traces"
	echo -e "\t--enable-driver      Enables protection kernel module (protection-agent) traces"

	if [[ $IS_X64_PROTECTION -eq 0 ]]; then
		echo -e "\t--enable-watchdog    Enables protection watchdog daemon (psanwatchdog) traces (not available on 32bit RHEL6.x/CentOS6.x/OL6.x distributions)"
		echo
	fi

	echo -e "\t--disable-all        Disables protection daemon, protection kernel module and watchdog traces"
	echo
	echo -e "\t--collect-logs       Gathers system and product information (also disables all traces)"
	echo

	exit 0
}




is_administrator


if [[ -z $1 ]]; then
	show_help
fi


for arg in $@
do
	case $arg in
		--collect-logs)
			gather_logs
			;;
		--enable-protection)
			enable_psanhostd_traces
			;;
		--enable-driver)
			enable_kernel_traces
			;;
		--enable-watchdog)
			enable_watchdog_traces
			;;
		--disable-all)
			disable_protection_traces
			;;
		--help)
			show_help
			;;
		--version)
			show_version
			;;
		*)
			show_help
			;;
	esac
done

